#!/bin/bash
#
#	Copyright © 2022 Claris International Inc.  All rights reserved.
#

LOG_DIR="/opt/FileMaker/FileMaker Server/Logs"

#
# Make sure gdb is installed. 
#
apt list gdb 2>&1 | grep installed > /dev/null 
RESULT=$?
if [[ $RESULT -ne 0 ]]; then
	echo "Debugger gdb is not installed."
	echo "Please run the command 'sudo apt install gdb' to install gdb..."
	exit 1
fi

#
# Make sure gdb is present
#
GDB_PATH=$(which gdb)
if [[ -z $GDB_PATH ]]; then
	echo "Error, gdb is not present after installation."
	exit 1
fi

#
# Make sure we have the privilege to attach to the process
#
if [[ $EUID -ne 0 ]]; then
	echo "Please run the script as 'sudo $0'"
	exit 1
fi

declare -a SERVER_PROCS=(fmserverd fmsased fmsib fmshelper fmscwpc fmwipd fmodata fmxdbc_listener)
declare -a THREAD_DUMPS

for proc in ${SERVER_PROCS[@]}; do
	procID=$(pgrep "$proc")
	if [[ -n "$procID" ]]; then
		TIME_STAMP=$(date +%b-%d-%H-%M-%S)
		LOG_FILE=${proc}_${TIME_STAMP}.log
		CMD_FILE="/tmp/gdbcmd_${procID}.txt"

		echo "Capturing thread information of $proc, process ID: $procID..."
#
# Generate gdb command files on the fly
#
		cat << EOF > "$CMD_FILE"
set pagination off	
set confirm off	
attach $procID
set logging off
set print pretty on
set logging file /tmp/$LOG_FILE
set logging on
thread apply all backtrace
detach
quit
EOF

#
# Direct console output to /dev/null to avoid cluttering
#
		"$GDB_PATH" -x "$CMD_FILE" > /dev/null 2>&1
		if [[  -f "/tmp/$LOG_FILE" ]]; then
			mv -f "/tmp/$LOG_FILE" "$LOG_DIR/$LOG_FILE" 
			chown fmserver:fmsadmin "$LOG_DIR/$LOG_FILE"
			THREAD_DUMPS+=("$LOG_FILE")
			rm -f "$CMD_FILE"
		else
			echo "GDB log file /tmp/$LOG_FILE is not generated"
		fi
	fi
done

if [[ ${#THREAD_DUMPS[@]} -gt 0 ]]; then
	echo "The following thread dumps are generated:"
	for log in ${THREAD_DUMPS[@]}; do
		echo "\"$LOG_DIR/$log\""
	done
else
	echo "No thread dumps are generated"
fi

